/******************************************************************************
 * Tab Converter
 *
 * (C) 2008 Mukunda Johnson
 ******************************************************************************/
 
#include <string>
#include <stdio.h>
#include <windows.h>

using namespace std;

enum {
	CHTAB	=0x09
};

enum ReturnValues {
	RETURN_USAGE,
	RETURN_OKAY,
	RETURN_INVALIDARG,
	RETURN_CANTOPENCLIPBOARD,
	RETURN_CANTALLOC,
	RETURN_INVALIDSOURCE,
};

const char *error_messages[] = {
	"Usage: tabconv <tabstop>\n",
	"",
	"Error: Invalid Tabstop.\n",
	"Error: Cannot open clipboard.\n",
	"Error: Cannot allocate memory for clipboard.\n",
	"Error: Clipboard does not contain text.\n"
};

#define PrintReturn(v) { printf( "%s", error_messages[v] ); return v; }

int main( int argc, char *argv[] ) {
	
	//-------------------------------------------------------------
	// Check for arguments
	//-------------------------------------------------------------
	if( argc < 2 )
		PrintReturn( RETURN_USAGE );
	
	//-------------------------------------------------------------
	// Read and verify tabstop
	//-------------------------------------------------------------
	int tabstop = atoi(argv[1]);
	if( tabstop == 0 )
		PrintReturn( RETURN_INVALIDARG );
	
	//-------------------------------------------------------------
	// Open clipboard for TEXT
	//-------------------------------------------------------------
	if( !IsClipboardFormatAvailable( CF_TEXT ) )
		PrintReturn( RETURN_INVALIDSOURCE );
	
	if( !OpenClipboard(NULL) )
		PrintReturn( RETURN_CANTOPENCLIPBOARD );
	
	//-------------------------------------------------------------
	// Get text pointer
	//-------------------------------------------------------------
	HGLOBAL hg = GetClipboardData( CF_TEXT );
	const char *source = (const char*)GlobalLock( hg );
	
	//-------------------------------------------------------------
	// Convert tabs to spaces according to tabstop
	//-------------------------------------------------------------
	string dest;
	
	int hpos = 0;
	
	for( int i = 0; source[i]; i++ ) {
	
		if( source[i] == CHTAB ) {
		
			// print spaces until next tab stop
			do {
				dest += ' ';
				hpos = (hpos + 1) % tabstop;
			} while( hpos != 0 );
			
		} else {
		
			dest += source[i];				//
			hpos = (hpos + 1) % tabstop;	//
			if( source[i] == '\r' ) {		// CR
				if( source[i+1] == '\n' ) {	// CR+LF
					i++;					// 
					dest += source[i];		//
				}							//
				hpos = 0;
				
			} else if( source[i] == '\n' ) {// LF
				
				hpos = 0;
			}
		}
	}
	
	GlobalUnlock( hg );
	
	//-------------------------------------------------------------
	// Allocate memory for clipboard
	//-------------------------------------------------------------
			
	HGLOBAL hCpy = GlobalAlloc( GMEM_MOVEABLE | GMEM_DDESHARE, dest.size()+1 );
	if( hCpy == NULL )
		PrintReturn( RETURN_CANTALLOC );
	
	//-------------------------------------------------------------
	// Copy new data to clipboard
	//-------------------------------------------------------------
	
	EmptyClipboard();
		
	char* ptr = (char*)GlobalLock( hCpy );
	
	const char *cdest = dest.c_str();
	
	int i;
	for( i = 0; cdest[i]; i++ ) {
		ptr[i] = cdest[i];
	}
	ptr[i] = 0;
	
	GlobalUnlock( hCpy );
	SetClipboardData( CF_TEXT, (HANDLE)hCpy );
	
	CloseClipboard();
	
	PrintReturn( RETURN_OKAY );
}
